'''
Created on 24.10.2017

    Class representing a cooling system, optimisation model generator. 
    Copyright (C) 2018  Miika Rama

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
	
@author: Miika Rama
'''

import oemof.solph as solph
from oemof import outputlib
import pandas as pd

from Classes.Producer import Producer
from Classes.Consumer import Consumer
from Classes.EnergySupply import EnergySupply
from Classes.EnergyStorage import EnergyStorage

#import csv
#from pandas._libs.period import Period
#from dill.dill import diff
import math
import time

class CoolingSystem(object):
    '''
    classdocs
    '''

    def __init__(self, name, master, tariff, charge):
        # for identification
        self.name = name

        # main components
        self.supply = []
        self.consumers = []
        self.producers = []
        self.network = []

        # for default properties for techs, resources
        self.mastersystem = master
        
        # cooling prices, �/MWh
        self.energyTariff = tariff
        
        # cooling prices, �/kW (max)
        self.capacityCharge = charge
        
        # use of resources
        self.usedResources = {}
        
        # system costs (�), results of the optimisation
        self.totalCosts = 0
        
        # storages, if capacity is 0 then no storage exists
        self.storages = {"dc storage": EnergyStorage(0,0,1/10,1/10, 0, "DC", 8),
                         "elec storage": EnergyStorage(0,0,1/10,1/10, 0, "Electricity", 0),
                         "heat storage": EnergyStorage(0,0,1/10,1/10, 0, "Heat", 50),
                         "cold storage": EnergyStorage(0,0,1/10,1/10, 0, "Cold", 12)}
        
        # period optimised
        self.period = 1
        
        # year optimised (in case resource prices change, 1 = original values used)
        self.year = 1
        
    def readDataFile(self, filename):
        try:
            f = open("Data/"+filename, 'r')
            data = f.readlines()
            for i in range(0,len(data[0].replace('\n', '').split('\t'))):                
                ts = []
                for j in range(0,len(data)):
                    line = data[j].replace('\n', '').split('\t')
                    value = line[i]
                    ts.append(float(value))
                self.consumers.append(Consumer(i+1, ts))
        except:
            print('Error reading file!!')
        finally:
            f.close()    

    def getAvailability(self, location, source):
        try:
            f = open("Data/availability_"+location+".txt", 'r')
            data = f.readlines()
            #items = data[0].replace('\n', '').split('\t')
            ts = []
            for i in range(1, len(data)):
                line = data[i].replace('\n', '').split('\t')
                if source == "Heat":
                    ts.append(float(line[0]))
                elif source == "Cold":
                    ts.append(float(line[1]))                   
                else:
                    ts.append(1.00)
                
            #print(len(ts))
            return(ts)
        except:
            print('Error reading file!!')
        finally:
            f.close()    
    
    def getNetworkLosses(self, Ts):
        if self.network == []:
            print("Network not defined...")
            return 0.0
        else:
            return self.network.calculateLosses()
    
    def getTotalEmissions(self, location):
        # calculate emission distribution per source, return total emissions
        totalEmissions = 0
        emissions = {}
        try:
            f = open("Data/emissions_"+location+".txt", 'r')
            data = f.readlines()
            items = data[0].replace('\n', '').split('\t')
            for i in range(1,len(items)):
                line1 = data[1].replace('\n', '').split('\t')
                line2 = data[2].replace('\n', '').split('\t')
                line3 = data[3].replace('\n', '').split('\t')
                # total emissions = consumption + production + plant (CHP/boiler) as g/kWh
                emissions[items[i]] = float(line1[i]) + float(line2[i]) + float(line3[i]) 

            for i in self.usedResources.keys():
                # kgCO2 = kWh * g/kWh / 1000
                totalEmissions += sum(self.usedResources[i]) * emissions[i] / 1000
        except:
            print('Error reading emissions file!!')
        finally:
            f.close()
        
        return totalEmissions
    
    def getEmbeddedEmissions(self):
        network = 0.0
        supply = 0.0
        cooling = 0.0
        storage = 0.0
        
        # network materials and constructions
        if self.network == []:
            print("Network not defined, zero embedded emissions...")
        else:
            network += self.network.calculateEmissions() 

        #print('Embedded emissions for network: '+str(network)+" kgCO2.")
        
        # energy supply specific embedded emissions (kgCO2/m2, NOW kgCO2/kW !!!!)
        try:
            f = open("Data/supply_techs.txt", 'r')
            
            # reading the file
            lines = f.readlines()
            
            supplyemissions = {}

            # disecting data 
            data = []
            for i in range(0, len(lines)):
                data.append(lines[i].replace('\n', '').split('\t'))
                                    
            for i in range(1, len(data[0])):
                supplyemissions[data[0][i]] = [float(data[7][i]), float(data[8][i])]
                
        except:
            print('Error reading file!!')
        finally:
            f.close() 
                
        # just PVs and SCs (combustion technologies included in the operational emissions)
        for i in range(0, len(self.supply)):
            if self.supply[i].capacity > 1000:
                supply += self.supply[i].capacity * supplyemissions[self.supply[i].tech][1] / self.supply[i].lifetime
            else:    
                supply += self.supply[i].capacity * supplyemissions[self.supply[i].tech][0] / self.supply[i].lifetime

        #print('Embedded emissions for supply: '+str(supply)+" kgCO2.")
          
        # cooling production specific embedded emissions (kgCO2/kW)
        try:
            f = open("Data/cooling_techs.txt", 'r')
            
            # reading the file
            lines = f.readlines()
            
            coolingemissions = {}

            # disecting data 
            data = []
            for i in range(0, len(lines)):
                data.append(lines[i].replace('\n', '').split('\t'))
                        
            for i in range(1, len(data[0])):
                coolingemissions[data[0][i]] = [float(data[8][i]), float(data[9][i])]
        except:
            print('Error reading file!!')
        finally:
            f.close()  
            
        for i in range(0, len(self.producers)):
            if self.producers[i].natr_eff > 0:
                if self.producers[i].capacity > 1000:
                    cooling += self.producers[i].capacity * coolingemissions['Free cooling'][1] / self.producers[i].lifetime
                else:
                    cooling += self.producers[i].capacity * coolingemissions['Free cooling'][0] / self.producers[i].lifetime
            elif self.producers[i].heat_eff > 0:
                if self.producers[i].capacity > 1000:
                    cooling += self.producers[i].capacity * coolingemissions['Absorption'][1] / self.producers[i].lifetime
                else:
                    cooling += self.producers[i].capacity * coolingemissions['Absorption'][0] / self.producers[i].lifetime
            else:
                if self.producers[i].capacity > 1000:
                    cooling += self.producers[i].capacity * coolingemissions['Compression'][1] / self.producers[i].lifetime
                else:
                    cooling += self.producers[i].capacity * coolingemissions['Compression'][0] / self.producers[i].lifetime
                        
        #print('Embedded emissions for cooling: '+str(cooling)+" kgCO2.")

        try:
            f = open("Data/storage_techs.txt", 'r')
            
            # reading the file
            lines = f.readlines()
            
            storage_emissions = {}

            # disecting data 
            data = []
            for i in range(0, len(lines)):
                data.append(lines[i].replace('\n', '').split('\t'))
                        
            for i in range(1, len(data[0])):
                # small storages, large storages, lifetime
                storage_emissions[data[0][i]] = [float(data[3][i]), float(data[4][i]), float(data[2][i])]
        except:
            print('Error reading file!!')
        finally:
            f.close()  
            
        for i in self.storages.keys():
            if self.storages[i].capacity > 0:
                self.storages[i].lifetime = storage_emissions[self.storages[i].tech][2]
                if self.storages[i].tech == "Electricity":
                    if self.storages[i].capacity > 1000:
                        storage += self.storages[i].capacity * storage_emissions[self.storages[i].tech][1] / self.storages[i].lifetime
                    else:
                        storage += self.storages[i].capacity * storage_emissions[self.storages[i].tech][0] / self.storages[i].lifetime
                # if not a battery storage, assuming a water based sensible heat storage
                else:
                    if self.storages[i].capacity > 1000:
                        # kWh * 3600 (=kJ) / (kJ/kgK * K) / 1000 kg/m3 * kgCO2/m3
                        storage += ( self.storages[i].capacity * 3600 / (4.18 * self.storages[i].dT) ) / 1000 * storage_emissions[self.storages[i].tech][1] / self.storages[i].lifetime
                    else:
                        storage += ( self.storages[i].capacity * 3600 / (4.18 * self.storages[i].dT) ) / 1000 * storage_emissions[self.storages[i].tech][0] / self.storages[i].lifetime
                    
        #print('Embedded emissions for storages: '+str(storage)+" kgCO2.")
        
        emissions = [network, supply, cooling, storage]
        
        return emissions
    
    def getTotalDemand(self):
        cooling_demand = self.getCoolingDemand()
        if self.network != []:
            for i in range(0,len(cooling_demand)):
                cooling_demand[i] += self.network.losses
        return cooling_demand
    
    def getCoolingDemand(self):
        '''
        Calculates the total cooling demand as the sum of cooling demands
        of individual, specified consumers.
        '''
        cooling_demand = []

        # checking if any consumers exist
        if len(self.consumers) == 0:
            print("No cooling demand data found!")
            return cooling_demand               
        
        # initialising the cooling demand as consumption of the first consumer
        for i in range(0,len(self.consumers[0].demand)):
            cooling_demand.append(self.consumers[0].demand[i])
        
        # summing the demands of (possible) other consumers with the initialised value
        if len(self.consumers) > 1:
            for i in range(1,len(self.consumers)): 
                for j in range(0,len(self.consumers[i].demand)): 
                    if self.consumers[i].demand[j] < 0:
                        cooling_demand[j] += 0
                    else:
                        cooling_demand[j] += self.consumers[i].demand[j]
        
        # checking if demand is zero (most likely an error)
        if sum(cooling_demand) == 0:
            print("Data found, but zero cooling demand.")
        
        # if running shorter periods than one year, cut the time series (affects summary results)
        if self.period < 8760:
            cooling_demand = cooling_demand[0:self.period]
        
        return cooling_demand
    
    def buildAndRun(self, period):           
        '''
        Builds and runs the optimisation model based of a defined system and
        processes the results, saving them into a .txt file.
        
        period    Optimisation period in hours (h)
        '''
        
        # *** model generation ***
        
        # setting up model
        datetimeindex = pd.date_range('1/1/2016', periods=period, freq='H')
        es = solph.EnergySystem(timeindex=datetimeindex)
                
        # preparation of input data (if needed)
        if self.network != []:
            self.network.calculateLosses()
        self.period = period
        cooling_demand = self.getCoolingDemand()
        
        print(str(len(self.consumers))+' consumers included!')        
        
        # defining buses from resources, energy supply and cooling production (hard-coded, unfortunately)
        b_elec = solph.Bus(label="electricity")   
        b_chips = solph.Bus(label="chips")
        b_pellets = solph.Bus(label="pellets")
        b_hfo = solph.Bus(label="hfo")
        b_lfo = solph.Bus(label="lfo")
        b_gas = solph.Bus(label="ng")
        b_solar = solph.Bus(label="sun")
        b_heat = solph.Bus(label="heat")
        b_cool = solph.Bus(label="cold") 
        b_dc = solph.Bus(label="district cooling")
        
        es.add(b_elec, b_chips, b_pellets, b_hfo, b_lfo, b_gas, b_solar, b_heat, b_cool, b_dc)
        
        # defining resources (Sources)  
        for i in range(0, len(self.mastersystem.resources)):
            if self.mastersystem.resources[i].name == 'Sun':    
                # Variable cost to 0; freely available, but production fixed (see energy supply). Feasibility issues?
                es.add(solph.Source(label=self.mastersystem.resources[i].name, 
                                    outputs={b_solar: solph.Flow(variable_costs=0)}))
            elif self.mastersystem.resources[i].name == 'HFO': 
                es.add(solph.Source(label=self.mastersystem.resources[i].name, 
                                    outputs={b_hfo: solph.Flow(variable_costs=[j*self.mastersystem.resources[i].projection[min(len(self.mastersystem.resources[i].projection), self.year)-1]/1000 for j in self.mastersystem.resources[i].availability])}))
            elif self.mastersystem.resources[i].name == 'LFO': 
                es.add(solph.Source(label=self.mastersystem.resources[i].name, 
                                    outputs={b_lfo: solph.Flow(variable_costs=[j*self.mastersystem.resources[i].projection[min(len(self.mastersystem.resources[i].projection), self.year)-1]/1000 for j in self.mastersystem.resources[i].availability])}))
            elif self.mastersystem.resources[i].name == 'NG': 
                es.add(solph.Source(label=self.mastersystem.resources[i].name, 
                                    outputs={b_gas: solph.Flow(variable_costs=[j*self.mastersystem.resources[i].projection[min(len(self.mastersystem.resources[i].projection), self.year)-1]/1000 for j in self.mastersystem.resources[i].availability])}))
            elif self.mastersystem.resources[i].name == 'Electricity': 
                es.add(solph.Source(label=self.mastersystem.resources[i].name, 
                                    outputs={b_elec: solph.Flow(variable_costs=[j*self.mastersystem.resources[i].projection[min(len(self.mastersystem.resources[i].projection), self.year)-1]/1000 for j in self.mastersystem.resources[i].availability])}))
            elif self.mastersystem.resources[i].name == 'Chips': 
                es.add(solph.Source(label=self.mastersystem.resources[i].name, 
                                    outputs={b_chips: solph.Flow(variable_costs=[j*self.mastersystem.resources[i].projection[min(len(self.mastersystem.resources[i].projection), self.year)-1]/1000 for j in self.mastersystem.resources[i].availability])}))
            elif self.mastersystem.resources[i].name == 'Pellets': 
                es.add(solph.Source(label=self.mastersystem.resources[i].name, 
                                    outputs={b_pellets: solph.Flow(variable_costs=[j*self.mastersystem.resources[i].projection[min(len(self.mastersystem.resources[i].projection), self.year)-1]/1000 for j in self.mastersystem.resources[i].availability])}))
            elif self.mastersystem.resources[i].name == 'Heat':
                es.add(solph.Source(label=self.mastersystem.resources[i].name, 
                                    outputs={b_heat: solph.Flow(variable_costs=[j*self.mastersystem.resources[i].projection[min(len(self.mastersystem.resources[i].projection), self.year)-1]/1000 for j in self.mastersystem.resources[i].availability])}))
            elif self.mastersystem.resources[i].name == 'Cold':
                es.add(solph.Source(label=self.mastersystem.resources[i].name, 
                                    outputs={b_cool: solph.Flow(variable_costs=[j*self.mastersystem.resources[i].projection[min(len(self.mastersystem.resources[i].projection), self.year)-1]/1000 for j in self.mastersystem.resources[i].availability])}))
        
        # all resources always defined           
        #print(str(len(self.mastersystem.resources))+' resources defined!')
        
        # defining energy supply (not always needed)
        
        for i in range(0,len(self.supply)):
            #print(self.supply[i].name+' with '+str(self.supply[i].capacity)+' kW in capacity and efficiencies (elec/heat) of '+
            #      str(self.supply[i].elec_eff)+'/'+str(self.supply[i].heat_eff))
            if self.supply[i].tech == "CHP":
                if self.supply[i].resource == "HFO":                    
                    es.add(solph.Transformer(label=self.supply[i].name, 
                                             inputs={b_hfo: solph.Flow()},
                                             outputs={b_elec: solph.Flow(nominal_value=self.supply[i].capacity/(self.supply[i].heat_eff+self.supply[i].elec_eff)*self.supply[i].elec_eff), 
                                                      b_heat: solph.Flow(nominal_value=self.supply[i].capacity/(self.supply[i].heat_eff+self.supply[i].elec_eff)*self.supply[i].heat_eff)},
                                             conversion_factors={b_elec: self.supply[i].elec_eff, b_heat: self.supply[i].heat_eff}))
                elif self.supply[i].resource == "LFO":    
                    es.add(solph.Transformer(label=self.supply[i].name, 
                                             inputs={b_lfo: solph.Flow()},
                                             outputs={b_elec: solph.Flow(nominal_value=self.supply[i].capacity/(self.supply[i].heat_eff+self.supply[i].elec_eff)*self.supply[i].elec_eff), 
                                                      b_heat: solph.Flow(nominal_value=self.supply[i].capacity/(self.supply[i].heat_eff+self.supply[i].elec_eff)*self.supply[i].heat_eff)},
                                             conversion_factors={b_elec: self.supply[i].elec_eff, b_heat: self.supply[i].heat_eff}))
                elif self.supply[i].resource == "NG":    
                    es.add(solph.Transformer(label=self.supply[i].name, 
                                             inputs={b_gas: solph.Flow()},
                                             outputs={b_elec: solph.Flow(nominal_value=self.supply[i].capacity/(self.supply[i].heat_eff+self.supply[i].elec_eff)*self.supply[i].elec_eff), 
                                                      b_heat: solph.Flow(nominal_value=self.supply[i].capacity/(self.supply[i].heat_eff+self.supply[i].elec_eff)*self.supply[i].heat_eff)},
                                             conversion_factors={b_elec: self.supply[i].elec_eff, b_heat: self.supply[i].heat_eff}))
                elif self.supply[i].resource == "Chips":    
                    es.add(solph.Transformer(label=self.supply[i].name, 
                                             inputs={b_chips: solph.Flow()},
                                             outputs={b_elec: solph.Flow(nominal_value=self.supply[i].capacity/(self.supply[i].heat_eff+self.supply[i].elec_eff)*self.supply[i].elec_eff), 
                                                      b_heat: solph.Flow(nominal_value=self.supply[i].capacity/(self.supply[i].heat_eff+self.supply[i].elec_eff)*self.supply[i].heat_eff)},
                                             conversion_factors={b_elec: self.supply[i].elec_eff, b_heat: self.supply[i].heat_eff}))
                elif self.supply[i].resource == "Pellets":    
                    es.add(solph.Transformer(label=self.supply[i].name, 
                                             inputs={b_pellets: solph.Flow()},
                                             outputs={b_elec: solph.Flow(nominal_value=self.supply[i].capacity/(self.supply[i].heat_eff+self.supply[i].elec_eff)*self.supply[i].elec_eff), 
                                                      b_heat: solph.Flow(nominal_value=self.supply[i].capacity/(self.supply[i].heat_eff+self.supply[i].elec_eff)*self.supply[i].heat_eff)},
                                             conversion_factors={b_elec: self.supply[i].elec_eff, b_heat: self.supply[i].heat_eff}))
                else:
                    print(self.supply[i].name+" resource ("+self.supply[i].resource+") not identified, supply not defined.")
            elif self.supply[i].tech == "Boiler":
                if self.supply[i].resource == "HFO":                    
                    es.add(solph.Transformer(label=self.supply[i].name, 
                                             inputs={b_hfo: solph.Flow()},
                                             outputs={b_heat: solph.Flow(nominal_value=self.supply[i].capacity)},
                                             conversion_factors={b_heat: self.supply[i].heat_eff}))
                elif self.supply[i].resource == "LFO":    
                    es.add(solph.Transformer(label=self.supply[i].name, 
                                             inputs={b_lfo: solph.Flow()},
                                             outputs={b_heat: solph.Flow(nominal_value=self.supply[i].capacity)},
                                             conversion_factors={b_heat: self.supply[i].heat_eff}))
                elif self.supply[i].resource == "NG":    
                    es.add(solph.Transformer(label=self.supply[i].name, 
                                             inputs={b_gas: solph.Flow()},
                                             outputs={b_heat: solph.Flow(nominal_value=self.supply[i].capacity)},
                                             conversion_factors={b_heat: self.supply[i].heat_eff}))
                elif self.supply[i].resource == "Chips":    
                    es.add(solph.Transformer(label=self.supply[i].name, 
                                             inputs={b_chips: solph.Flow()},
                                             outputs={b_heat: solph.Flow(nominal_value=self.supply[i].capacity)},
                                             conversion_factors={b_heat: self.supply[i].heat_eff}))
                elif self.supply[i].resource == "Pellets":    
                    es.add(solph.Transformer(label=self.supply[i].name, 
                                             inputs={b_pellets: solph.Flow()},
                                             outputs={b_heat: solph.Flow(nominal_value=self.supply[i].capacity)},
                                             conversion_factors={b_heat: self.supply[i].heat_eff}))
                else:
                    print(self.supply[i].name+" resource ("+self.supply[i].resource+") not identified, supply not defined.")
            
            elif self.supply[i].tech == "PVs":
                es.add(solph.Transformer(label=self.supply[i].name, 
                                         inputs={b_solar: solph.Flow()},
                                         outputs={b_elec: solph.Flow(actual_value=self.calculateSolarPhotovoltaicOutput(period, self.mastersystem.location, self.supply[i].capacity), 
                                                                     nominal_value=1, fixed=True)},
                                         conversion_factors={b_elec: 1.0}))
            elif self.supply[i].tech == "Collectors":
                es.add(solph.Transformer(label=self.supply[i].name, 
                                         inputs={b_solar: solph.Flow()},
                                         outputs={b_heat: solph.Flow(actual_value=self.calculateSolarCollectorOutput(period, self.mastersystem.location, self.supply[i].capacity), 
                                                                     nominal_value=1, fixed=True)},
                                         conversion_factors={b_heat: 1.0}))
            else:
                print("Unknown supply technology ("+self.supply[i].tech+"), supply not defined.")
                
        print(str(len(self.supply))+' points of supply defined!')    

        # cooling production
        for i in range(0,len(self.producers)):
            #print(self.producers[i].name+' with '+str(self.producers[i].capacity)+' kW in capacity, efficiencies (elec/heat/nat) of '+
            #      str(self.producers[i].elec_eff)+'/'+str(self.producers[i].heat_eff)+'/'+str(self.producers[i].natr_eff))
            if self.producers[i].natr_eff > 0:
                if self.producers[i].elec_eff > 0:                
                    ava = self.getAvailability(self.mastersystem.location, "Cold")                   
                    es.add(solph.Transformer(label=self.producers[i].name, 
                                      inputs={b_elec: solph.Flow(), b_cool: solph.Flow()}, 
                                      outputs={b_dc: solph.Flow(nominal_value=self.producers[i].capacity, min=[0.0]*8760, max=ava)}, 
                                      conversion_factors={b_elec: self.producers[i].elec_eff,
                                                          b_cool: self.producers[i].natr_eff}))
                
                else:
                    es.add(solph.Transformer(label=self.producers[i].name, 
                                      inputs={b_cool: solph.Flow()}, 
                                      outputs={b_dc: solph.Flow(nominal_value=self.producers[i].capacity)}, 
                                      conversion_factors={b_cool: self.producers[i].natr_eff}))                   
            elif self.producers[i].heat_eff > 0:    
                if self.producers[i].elec_eff > 0:                
                    es.add(solph.Transformer(label=self.producers[i].name, 
                                      inputs={b_elec: solph.Flow(), b_heat: solph.Flow()}, 
                                      outputs={b_dc: solph.Flow(nominal_value=self.producers[i].capacity)}, 
                                      conversion_factors={b_elec: self.producers[i].elec_eff,
                                                          b_heat: self.producers[i].heat_eff}))
                else:
                    es.add(solph.Transformer(label=self.producers[i].name, 
                                      inputs={b_heat: solph.Flow()}, 
                                      outputs={b_dc: solph.Flow(nominal_value=self.producers[i].capacity)}, 
                                      conversion_factors={b_heat: self.producers[i].heat_eff}))
                    
            else:
                es.add(solph.Transformer(label=self.producers[i].name, 
                                  inputs={b_elec: solph.Flow()}, 
                                  outputs={b_dc: solph.Flow(nominal_value=self.producers[i].capacity)}, 
                                  conversion_factors={b_elec: self.producers[i].elec_eff}))
        
        print(str(len(self.producers))+' points of cooling defined!')    
               
        # storage definition
        if self.storages['dc storage'].capacity > 0:
            es.add(solph.components.GenericStorage(label='dc storage', inputs={b_dc: solph.Flow(variable_costs=self.storages['dc storage'].loadingCost)}, 
            outputs={b_dc: solph.Flow(variable_costs=self.storages['dc storage'].unloadingCost)}, 
            capacity_loss=self.storages['dc storage'].losses, 
            nominal_capacity=self.storages['dc storage'].capacity, 
            initial_capacity=self.storages['dc storage'].initCharge, 
            nominal_input_capacity_ratio=self.storages['dc storage'].loadingRatio, 
            nominal_output_capacity_ratio=self.storages['dc storage'].unloadingRatio,
            inflow_conversion_factor=self.storages['dc storage'].loadingEff, 
            outflow_conversion_factor=self.storages['dc storage'].unloadingEff))   

        if self.storages['elec storage'].capacity > 0:
            es.add(solph.components.GenericStorage(label='elec storage', inputs={b_elec: solph.Flow(variable_costs=self.storages['elec storage'].loadingCost)}, 
            outputs={b_dc: solph.Flow(variable_costs=self.storages['elec storage'].unloadingCost)}, 
            capacity_loss=self.storages['elec storage'].losses, 
            nominal_capacity=self.storages['elec storage'].capacity, 
            initial_capacity=self.storages['elec storage'].initCharge, 
            nominal_input_capacity_ratio=self.storages['elec storage'].loadingRatio, 
            nominal_output_capacity_ratio=self.storages['elec storage'].unloadingRatio,
            inflow_conversion_factor=self.storages['elec storage'].loadingEff, 
            outflow_conversion_factor=self.storages['elec storage'].unloadingEff))   

        if self.storages['heat storage'].capacity > 0:
            es.add(solph.components.GenericStorage(label='heat storage', inputs={b_elec: solph.Flow(variable_costs=self.storages['heat storage'].loadingCost)}, 
            outputs={b_dc: solph.Flow(variable_costs=self.storages['heat storage'].unloadingCost)}, 
            capacity_loss=self.storages['heat storage'].losses, 
            nominal_capacity=self.storages['heat storage'].capacity, 
            initial_capacity=self.storages['heat storage'].initCharge, 
            nominal_input_capacity_ratio=self.storages['heat storage'].loadingRatio, 
            nominal_output_capacity_ratio=self.storages['heat storage'].unloadingRatio,
            inflow_conversion_factor=self.storages['heat storage'].loadingEff, 
            outflow_conversion_factor=self.storages['heat storage'].unloadingEff))   
        
        if self.storages['cold storage'].capacity > 0:
            es.add(solph.components.GenericStorage(label='cold storage', inputs={b_elec: solph.Flow(variable_costs=self.storages['cold storage'].loadingCost)}, 
            outputs={b_dc: solph.Flow(variable_costs=self.storages['cold storage'].unloadingCost)}, 
            capacity_loss=self.storages['cold storage'].losses, 
            nominal_capacity=self.storages['cold storage'].capacity, 
            initial_capacity=self.storages['cold storage'].initCharge, 
            nominal_input_capacity_ratio=self.storages['cold storage'].loadingRatio, 
            nominal_output_capacity_ratio=self.storages['cold storage'].unloadingRatio,
            inflow_conversion_factor=self.storages['cold storage'].loadingEff, 
            outflow_conversion_factor=self.storages['cold storage'].unloadingEff))   

        # distribution network (losses), HEY: MAYBE PUMPING COULD BE PUT HERE?? 
        if self.network != []:
            es.add(solph.Sink(label="network losses", inputs={b_dc: solph.Flow(actual_value=self.network.losses, nominal_value=1, fixed=True)}))
        
        # cooling demand
        es.add(solph.Sink(label="cooling demand", inputs={b_dc: solph.Flow(actual_value=cooling_demand, nominal_value=1, fixed=True)}))
        
        # pumping electricity consumption
        if self.network != []:
            pumping = self.network.calculatePumpingElectricityConsumption(self.getTotalDemand())
            es.add(solph.Sink(label="pumping", inputs={b_elec: solph.Flow(actual_value=pumping, nominal_value=1, fixed=True)}))
               
        # sinks for electricity and heating (there could actually be two for each; for market and for feasibility)       
        es.add(solph.Sink(label="sold electricity", inputs={b_elec: solph.Flow(variable_cost=-0.01)}))
        es.add(solph.Sink(label="sold heat", inputs={b_heat: solph.Flow(variable_cost=-0.001)}))
        
        print("Model components defined, building the model... ", end="", flush=True)
        
        # generating the model
        om = solph.Model(es)
        
        print("Done!")
        
        # *** optimisation model solving ***
        
        # solve with specific optimization options (passed to pyomo)
        #om.solve(solver='cbc', solve_kwargs={'tee': True, 'keepfiles': True})
        print('Solving... ', end="", flush=True)
        #om.solve(solver='cbc',solve_kwargs={'tee': True},cmdline_options = {'sec': '120'})
        outcome = om.solve(solver='cbc',cmdline_options = {'sec': '120'})
        
        if outcome["Solver"][0]["Status"].key != "ok" or outcome["Solver"][0]["Termination condition"].key != "optimal":
            print('Optimal solution not found!')
            return 0
        else:
            print('Done!')
        
        # *** results processing ***
        
        print("Processing results... ", end="", flush=True)
        
        # write back results from optimisation 
        res = outputlib.processing.results(om)

        # print value of objective function
        meta_results = outputlib.processing.meta_results(om)
        self.totalCosts = meta_results['objective']
                        
        # gets all resource related results
        resources = {}
        for i in range(0, len(self.mastersystem.resources)):
            data = outputlib.views.node(res, self.mastersystem.resources[i].name)['sequences'][((self.mastersystem.resources[i].name, self.mastersystem.resources[i].name.lower()), 'flow')][:]
            resources[self.mastersystem.resources[i].name] = data
            
            # save results on used resources for economic calculation
            if (sum(data)>0):
                self.usedResources[self.mastersystem.resources[i].name] = data

        # energy supply specific results
        energysupply = {}
        for i in range(0,len(self.supply)):
            if self.supply[i].tech == 'CHP':
                data1 = outputlib.views.node(res, self.supply[i].name)['sequences'][((self.supply[i].resource.lower(), self.supply[i].name), 'flow')][:]
                data2 = outputlib.views.node(res, self.supply[i].name)['sequences'][((self.supply[i].name, "electricity"), 'flow')][:]
                data3 = outputlib.views.node(res, self.supply[i].name)['sequences'][((self.supply[i].name, "heat"), 'flow')][:]
                energysupply[self.supply[i].name+' ('+self.supply[i].resource+')'] = data1
                energysupply[self.supply[i].name+' (elec)'] = data2
                energysupply[self.supply[i].name+' (heat)'] = data3
                self.supply[i].consumption = data1
                self.supply[i].elec_production = data2
                self.supply[i].heat_production = data3
                self.supply[i].utilisation = (sum(self.supply[i].elec_production) + sum(self.supply[i].heat_production)) / (self.supply[i].capacity * period)                    
            elif self.supply[i].tech == 'Boiler':
                data1 = outputlib.views.node(res, self.supply[i].name)['sequences'][((self.supply[i].resource.lower(), self.supply[i].name), 'flow')][:]
                data2 = outputlib.views.node(res, self.supply[i].name)['sequences'][((self.supply[i].name, "heat"), 'flow')][:]
                energysupply[self.supply[i].name+' ('+self.supply[i].resource+')'] = data1
                energysupply[self.supply[i].name+' (heat)'] = data2
                self.supply[i].consumption = data1
                self.supply[i].elec_production = [0] * period
                self.supply[i].heat_production = data2                    
                self.supply[i].utilisation = (sum(self.supply[i].heat_production)) / (self.supply[i].capacity * period)                    
            elif self.supply[i].tech == 'Collectors':
                data = outputlib.views.node(res, self.supply[i].name)['sequences'][((self.supply[i].name, "heat"), 'flow')][:]
                self.supply[i].elec_production = [0] * period
                self.supply[i].heat_production = data
                energysupply[self.supply[i].name] = data
                # capapacity given as m2, utilisation rate calculated using peak production
                self.supply[i].utilisation = (sum(self.supply[i].heat_production)) / (max(self.supply[i].heat_production) * period)                    
            elif self.supply[i].tech == 'PVs':
                data = data.append(outputlib.views.node(res, self.supply[i].name)['sequences'][((self.supply[i].name, "electricity"), 'flow')][:])                                
                self.supply[i].elec_production = data
                self.supply[i].heat_production = [0] * period
                energysupply[self.supply[i].name] = data
                # capapacity given as m2, utilisation rate calculated using peak production
                self.supply[i].utilisation = (sum(self.supply[i].elec_production)) / (max(self.supply[i].elec_production) * period)                    
            else:
                print('WARNING: An unknown energy supply technology.')        
            
        # cooling production
        producers = {}
        for i in range(0,len(self.producers)):
            data = outputlib.views.node(res, self.producers[i].name)['sequences'][((self.producers[i].name, "district cooling"), 'flow')][:]
            producers[self.producers[i].name+' (dc)'] = data
            self.producers[i].production = data
            self.producers[i].utilisation = sum(self.producers[i].production) / (self.producers[i].capacity * period) 
            
            if self.producers[i].elec_eff > 0:
                data1 = outputlib.views.node(res, self.producers[i].name)['sequences'][(("electricity", self.producers[i].name), 'flow')][:]                          
                producers[self.producers[i].name+' (elec)'] = data1
                self.producers[i].elec_cons = data1  
            
            if self.producers[i].heat_eff > 0:
                data2 = outputlib.views.node(res, self.producers[i].name)['sequences'][(("heat", self.producers[i].name), 'flow')][:]                          
                producers[self.producers[i].name+' (heat)'] = data2  
                self.producers[i].heat_cons = data2  
            
            if self.producers[i].natr_eff > 0:
                data3 = outputlib.views.node(res, self.producers[i].name)['sequences'][(("cold", self.producers[i].name), 'flow')][:]                          
                producers[self.producers[i].name+' (natr)'] = data3
                self.producers[i].natr_cons = data3  

        # demand and losses
        if self.storages["dc storage"].capacity > 0:
            load = outputlib.views.node(res, "dc storage")['sequences'][(("district cooling", "dc storage"), 'flow')][:]
            unload = outputlib.views.node(res, "dc storage")['sequences'][(("dc storage", "district cooling"), 'flow')][:]
            self.storages["dc storage"].load = load
            self.storages["dc storage"].unload = unload
            self.storages["dc storage"].cycles = sum(self.storages["dc storage"].load) / (self.storages["dc storage"].capacity)
        if self.storages["elec storage"].capacity > 0:
            load = outputlib.views.node(res, "elec storage")['sequences'][(("electricity", "elec storage"), 'flow')][:]
            unload = outputlib.views.node(res, "elec storage")['sequences'][(("elec storage", "electricity"), 'flow')][:]
            self.storages["elec storage"].load = load
            self.storages["elec storage"].unload = unload
            self.storages["elec storage"].cycles = sum(self.storages["elec storage"].load) / (self.storages["elec storage"].capacity)
        if self.storages["heat storage"].capacity > 0:
            load = outputlib.views.node(res, "heat storage")['sequences'][(("heat", "heat storage"), 'flow')][:]
            unload = outputlib.views.node(res, "heat storage")['sequences'][(("heat storage", "heat"), 'flow')][:]
            self.storages["heat storage"].load = load
            self.storages["heat storage"].unload = unload
            self.storages["heat storage"].cycles = sum(self.storages["heat storage"].load) / (self.storages["heat storage"].capacity)
        if self.storages["cold storage"].capacity > 0:
            load = outputlib.views.node(res, "cold storage")['sequences'][(("cold", "cold storage"), 'flow')][:]
            unload = outputlib.views.node(res, "cold storage")['sequences'][(("cold storage", "cold"), 'flow')][:]
            self.storages["cold storage"].load = load
            self.storages["cold storage"].unload = unload
            self.storages["cold storage"].cycles = sum(self.storages["cold storage"].load) / (self.storages["cold storage"].capacity)

        # cooling demand
        demand = outputlib.views.node(res, "cooling demand")['sequences'][(("district cooling", "cooling demand"), 'flow')][:]

        
        # network losses and pumping consumption, if a network exists
        if self.network == []:
            pumpcons = [0] * period
            losses = [0] * period
        else:    
            losses = outputlib.views.node(res, "network losses")['sequences'][(("district cooling", "network losses"), 'flow')][:]
            pumpcons = outputlib.views.node(res, "pumping")['sequences'][(("electricity", "pumping"), 'flow')][:]

        print("Done!")

        # opening a text file for results
        thefile = open('Optimisation results ('+self.name+').txt', 'w')
                
        # result time series titles, starting from resources..
        sorted_resources = sorted(resources)
        for i in range(0,len(sorted_resources)):
            thefile.write("%s;" % sorted_resources[i])            
        # then energy supply..
        sorted_supply = sorted(energysupply)
        for i in range(0,len(sorted_supply)):
            thefile.write("%s;" % sorted_supply[i])            
        # then production supply..
        sorted_producers = sorted(producers)
        for i in range(0,len(sorted_producers)):
            thefile.write("%s;" % sorted_producers[i])            
        # finally demand, losses and storage (load, unload)
        thefile.write("%s;%s;%s" % ('Demand', 'Losses', 'Pumping'))
        if self.storages["dc storage"].capacity > 0:
            thefile.write(";%s;%s" % ('DC loading', 'DC unloading'))            
        if self.storages["elec storage"].capacity > 0:
            thefile.write(";%s;%s" % ('Elec loading', 'Elec unloading'))            
        if self.storages["heat storage"].capacity > 0:
            thefile.write(";%s;%s" % ('Heat loading', 'Heat unloading'))            
        if self.storages["cold storage"].capacity > 0:
            thefile.write(";%s;%s" % ('Cold loading', 'Cold unloading'))            
        thefile.write("\n")
  
        # writing result time series themselves
        for i in range(0, period):
            for j in range(0,len(sorted_resources)):
                thefile.write("%r;" % resources[sorted_resources[j]][i])
            for j in range(0,len(sorted_supply)):
                thefile.write("%r;" % energysupply[sorted_supply[j]][i])
            for j in range(0,len(sorted_producers)):
                thefile.write("%r;" % producers[sorted_producers[j]][i])   
            thefile.write("%r;%r;%r" % (demand[i], losses[i], pumpcons[i]))
            if self.storages["dc storage"].capacity > 0:
                thefile.write(";%r;%r" % (self.storages["dc storage"].load[i], self.storages["dc storage"].unload[i]))
            if self.storages["elec storage"].capacity > 0:
                thefile.write(";%r;%r" % (self.storages["elec storage"].load[i], self.storages["elec storage"].unload[i]))
            if self.storages["heat storage"].capacity > 0:
                thefile.write(";%r;%r" % (self.storages["heat storage"].load[i], self.storages["heat storage"].unload[i]))
            if self.storages["cold storage"].capacity > 0:
                thefile.write(";%r;%r" % (self.storages["cold storage"].load[i], self.storages["cold storage"].unload[i]))
            thefile.write("\n")

        thefile.close()
        print("Results file written.")
        return 1
                
    def getRadiationData(self, location):
        # get radiation data from a separate file (country specific)        
        try:
            f = open("Data/solar_"+location+".txt", 'r')
            # diffuse, direct, temperature
            ddt = []
            data = f.readlines()
            for i in range(1,len(data)):
                line = data[i].replace('\n', '').split('\t')
                ddt.append([float(line[1]), float(line[0]), float(line[2])])
        except:
            print('Error reading solar irradation file!!')
        finally:
            f.close()
            
        return ddt
    
    def calculateSolarPhotovoltaicOutput(self, period, location, area):
        # get outdoor temperature (Ta) and radiation data (G_bn, G_dh, G_th)
        ddt = self.getRadiationData(location)
        
        # calculate output based on PV specification (area), A VERY SIMPLIFIED VERSION!   
        
        # fixed values
        KM = 0.8
        
        # fixed efficiency for PV
        n0 = 0.13
                        
        # Incident angle between the solar beam and the normal to the collector
        phi = math.pi/8
        
        # collector tilt angle
        beta = math.pi/6
        
        # ground reflection (typically 0.1-0.3)
        roo = 0.2
        
        # uncertainty factors; heat losses, measurement uncertainty..
        fouling = 0.99
        
        # not needed?
        # n_sc =   P_sc/GS = n_0 -  (a_1 *(T_f-T_a))/G - (a_2* (T_f-T_a)^2)/G

        P = [0]*period 
        for i in range(0, period):        
            G = (ddt[i][1]*math.cos(phi)+ ddt[i][0]*(1+math.cos(beta))/2+ roo*(ddt[i][0]+ddt[i][1])*(1-math.cos(beta))/2 ) * KM
            #G=G0 * KM   
            # solar collector output in kW
            P[i]= max(0,(n0*G/1000*fouling*area))
            #print(P[i])
             
        return P
    
    def calculateSolarCollectorOutput(self, period, location, area):
        
        # get outdoor temperature (Ta) and radiation data (G_bn, G_dh, G_th)
        ddt = self.getRadiationData(location)
        
        # calculate output based on collector specification (area)   
        
        # fixed values
        KM = 0.8
        n0 = 0.85
        a1 = 2.3 # W/m2K
        a2 = 0.029 # W/m2K 
        
        # mean temperature of the collector fluid
        Tf = 60
        
        # Incident angle between the solar beam and the normal to the collector
        phi = math.pi/8
        
        # collector tilt angle
        beta = math.pi/6
        
        # ground reflection (typically 0.1-0.3)
        roo = 0.2
        
        # uncertainty factors; heat losses, measurement uncertainty..
        fp = 1
        fo = 1
        fu = 1
        
        # not needed?
        # n_sc =   P_sc/GS = n_0 -  (a_1 *(T_f-T_a))/G - (a_2* (T_f-T_a)^2)/G

        Q = [0]*period 
        for i in range(0, period):        
            G = (ddt[i][1]*math.cos(phi)+ ddt[i][0]*(1+math.cos(beta))/2+ roo*(ddt[i][0]+ddt[i][1])*(1-math.cos(beta))/2 ) * KM
            #G=G0 * KM   
            # solar collector output in kW
            Q[i]= max(0,(n0*G-a1*(Tf-ddt[i][2])-a2*(Tf-ddt[i][2])**2)*area*fp*fo*fu/1000) 
            #print(Q[i])
             
        return Q